import { readCliMetadata } from '../cli-metadata.js';
import { expectValue } from './flag-utils.js';
import { buildGenerateCliCommand, shellQuote } from './generate/template-data.js';
import { formatSourceSuffix } from './list-format.js';
import { consumeOutputFormat } from './output-format.js';
import { formatPathForDisplay } from './path-utils.js';
export async function handleInspectCli(args) {
    const parsed = parseInspectFlags(args);
    const metadata = await readCliMetadata(parsed.artifactPath);
    if (parsed.format === 'json') {
        console.log(JSON.stringify(metadata, null, 2));
        return;
    }
    console.log(`Artifact: ${formatPathForDisplay(metadata.artifact.path)} (${metadata.artifact.kind})`);
    console.log(`Server: ${metadata.server.name}`);
    if (metadata.server.source) {
        const suffix = formatSourceSuffix(metadata.server.source, true);
        if (suffix) {
            console.log(`Source: ${suffix}`);
        }
    }
    console.log(`Generated: ${new Date(metadata.generatedAt).toISOString()} via ${metadata.generator.name}@${metadata.generator.version}`);
    if (metadata.invocation.runtime) {
        console.log(`Runtime: ${metadata.invocation.runtime}`);
    }
    console.log('Invocation flags:');
    for (const [key, value] of Object.entries(metadata.invocation)) {
        if (value === undefined || value === null || key === 'runtime') {
            continue;
        }
        console.log(`  ${key}: ${Array.isArray(value) ? JSON.stringify(value) : String(value)}`);
    }
    const dryRunCommand = buildGenerateCliCommand(metadata.invocation, metadata.server.definition);
    console.log('Regenerate with:');
    console.log(`  mcporter generate-cli --from ${shellQuote(parsed.artifactPath)}`);
    if (dryRunCommand) {
        console.log('Underlying generate-cli command:');
        console.log(`  ${dryRunCommand}`);
    }
}
function parseInspectFlags(args) {
    let format = consumeOutputFormat(args, {
        defaultFormat: 'text',
        allowed: ['text', 'json'],
        enableRawShortcut: false,
        jsonShortcutFlag: '--json',
    });
    let index = 0;
    while (index < args.length) {
        const token = args[index];
        if (!token) {
            index += 1;
            continue;
        }
        if (token === '--format') {
            const value = expectValue(token, args[index + 1]);
            if (value !== 'json' && value !== 'text') {
                throw new Error("--format must be 'json' or 'text'.");
            }
            format = value;
            args.splice(index, 2);
            continue;
        }
        if (token.startsWith('--')) {
            throw new Error(`Unknown flag '${token}' for inspect-cli.`);
        }
        index += 1;
    }
    const artifactPath = args.shift();
    if (!artifactPath) {
        throw new Error('Usage: mcporter inspect-cli <artifact> [--json]');
    }
    return { artifactPath, format };
}
export const __test = {
    parseInspectFlags,
};
//# sourceMappingURL=inspect-cli-command.js.map